﻿/* Copyright 2015 Intellica Corporation 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at
    http://www.apache.org/licenses/LICENSE-2.0
Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/
using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Data;
using System.Configuration;
using DataAccess;
using System.Web.SessionState;

using MDWSLib.MDWSEmrSvc;


    /// <summary>
    /// CMDWSOps is used to access MDWS operations and move data to the central database
    /// It uses DataTrasnferLib to move data
    /// </summary>
    public class CMDWSOps : CData
    {
        //Constructor
        public CMDWSOps(CData data)
            : base(data)
        {
            //constructors are not inherited in c#!
        }

        /// <summary>
        /// Is a cosigner required for this note title/user
        /// </summary>
        /// <param name="strNoteTitleIEN"></param>
        /// <param name="strAuthorDUZ"></param>
        /// <returns></returns>
        public bool IsCosignerRequired(string strNoteTitleIEN, string strAuthorDUZ)
        {
            //check to make sure the MDWS connection is valid
            /*CStatus status = IsMDWSValid();
            if (!status.Status)
            {
                return false;
            }
            */
            CStatus status = new CStatus();


            //get data from MDWS
            TextTO tto = GetMDWSSOAPClient().isCosignerRequired(strNoteTitleIEN, strAuthorDUZ);
            if (tto != null)
            {
                if (tto.text != null)
                {
                    if (tto.text.Trim().ToUpper() == "Y")
                    {
                        return true;
                    }
                }
            }

            return false;
        }

        /// <summary>
        /// US:5716 transfer a patient by DFN
        /// </summary>
        /// <param name="strkey"></param>
        /// <param name="strID"></param>
        /// <param name="lCount"></param>
        /// <returns></returns>
        public CStatus TransferPatientByDFN(string strkey,
                                           string strDFN,
                                           out string strPatID)
        {
            //central db patient id
            strPatID = string.Empty;

            //check to make sure the MDWS connection is valid
            /*CStatus status = IsMDWSValid();
            if (!status.Status)
            {
                return false;
            }
            */
            CStatus status = new CStatus();


            //get data from MDWS
            PatientTO toPat = GetMDWSSOAPClient().select(strDFN);
            if (toPat == null || toPat.fault != null)
            {
                string strM = "MDWS Patient lookup failed, if this error continues please contact your system administrator.";
                
                //per wasa scan do not return sensitive information to the user.
                //if (toPat.fault != null)
                //{
                //    strM = toPat.fault.message;
                //}
                return new CStatus(false, k_STATUS_CODE.Failed, strM);
            }

            CMDWSTransfer transfer = new CMDWSTransfer(this);
            status = transfer.TransferPatient(WebSession,
                                              strkey,
                                              toPat,
                                              out strPatID);

            return status;
        }

        /// <summary>
        /// US:5729 US:5349 tranfer MDWS patients that match criteria
        /// </summary>
        /// <param name="strMatchID"></param>
        /// <param name="lCount"></param>
        /// <returns></returns>
        public CStatus TransferMatchPatients(string strkey,
                                             long lMatchType,
                                             string strMatch,
                                             out long lCount)
        {
            //count
            lCount = 0;
         
            //check to make sure the MDWS connection is valid
            /*CStatus status = IsMDWSValid();
             if (!status.Status)
             {
                 return false;
             }
             */
            CStatus status = new CStatus();

            //get data from MDWS
            TaggedPatientArrays tpas = GetMDWSSOAPClient().match(strMatch);
            if (tpas == null || tpas.fault != null)
            {
                string strM = "MDWS Patient lookup failed, if this error continues please contact your system administrator.";

                //per wasa scan do not return sensitive information to the user.
                //if(tpas.fault != null)
                // {
                //    strM = tpas.fault.message;
                //}

                return new CStatus(false, k_STATUS_CODE.Failed, strM);
            }

            foreach (TaggedPatientArray tpa in tpas.arrays)
            {
                CMDWSTransfer transfer = new CMDWSTransfer(this);
                status = transfer.TransferPatientArray(WebSession,
                                                       tpa,
                                                       strkey,
                                                       lMatchType,
                                                       strMatch,
                                                       out lCount);
                if (!status.Status)
                {
                    return status;
                }
            }

            return new CStatus();
        }

        protected EmrSvcSoapClient m_EmrSvcSoapClient = null;
        /// <summary>
        /// gets the soap client so we can talk to mdws
        /// </summary>
        /// <returns></returns>
        protected EmrSvcSoapClient GetMDWSSOAPClient()
        {
            m_EmrSvcSoapClient = new EmrSvcSoapClient("EmrSvcSoap");
            return m_EmrSvcSoapClient;

            //this is used from communicator also and we 
            //have no "Session" in communicator
         /*   if (WebSession != null)
            {
                if (WebSession["EmrSvcSoapClient"] == null)
                {
                    m_EmrSvcSoapClient = new EmrSvcSoapClient("EmrSvcSoap");
                    WebSession["EmrSvcSoapClient"] = m_EmrSvcSoapClient;
                }

                return (EmrSvcSoapClient)WebSession["EmrSvcSoapClient"];
            }
            else
            {
                if (m_EmrSvcSoapClient == null)
                {
                    m_EmrSvcSoapClient = new EmrSvcSoapClient("EmrSvcSoap");
                }

                return m_EmrSvcSoapClient;
            }*/
        }

        /// <summary>
        /// US:840
        /// helper to determine if MDWS connection is valid  
        /// </summary>
        /// <returns></returns>
        public CStatus IsMDWSValid()
        {
            string strConnError = "You have lost connection to MDWS/CPRS, please login and try again!";
            CStatus statusFailed = new CStatus();
            statusFailed.StatusCode = k_STATUS_CODE.Failed;
            statusFailed.StatusComment = strConnError;
            statusFailed.Status = false;

            //fail if no soap client
            if (GetMDWSSOAPClient() == null)
            {
                return statusFailed;
            }

            //fail if no site id
            if (this.SiteID < 1)
            {
                return statusFailed;
            }

            //perform an operation that goes "deep" enough to keep the MDWS connection open
            //choosing getNoteTitles with search text that should never match
            //so the return is fast
            TaggedTextArray tta = GetMDWSSOAPClient().getNoteTitles("ZZZZ", "1");
            if (tta.fault != null)
            {
                //call failed
                return statusFailed;
            }

            //good to go
            return new CStatus();
        }

        /// <summary>
        /// disconnect from MDWs
        /// </summary>
        public void Disconnect()
        {
            GetMDWSSOAPClient().disconnect();
        }

        /// <summary>
        /// get all MDWS sites by calling getVHA
        /// </summary>
        /// <param name="ra"></param>
        /// <returns></returns>
        public CStatus TransferSites()
        {
            //do not check to see if we are connected to MDWS
            //a connection is not required for getVHA

            CStatus status = new CStatus();

            //get the sites from MDWS
            RegionArray ra = GetMDWSSOAPClient().getVHA();
            if (ra == null || ra.fault != null)
            {
                return new CStatus(false, k_STATUS_CODE.Failed, "getVHA failed!");
            }

            //transfer the sites to the central db
            CMDWSTransfer transfer = new CMDWSTransfer(this);
            status = transfer.TransferSites(ra);
            if (!status.Status)
            {
                return status;
            }

            return status;
        }


        /// <summary>
        /// protectd helper to get the next user target
        /// </summary>
        /// <param name="ua"></param>
        /// <param name="strTarget"></param>
        /// <returns></returns>
        protected CStatus GetNextUserTarget(UserArray ua, out string strTarget)
        {
            strTarget = String.Empty;

            if (ua.users == null)
            {
                return new CStatus();
            }

            //get the index of the last item in the array
            int nCountArray = ua.count;
            int nIndexArray = 0;
            if (nCountArray > 0)
            {
                nIndexArray = nCountArray - 1;
            }

            //get the next target
            strTarget = ua.users[nIndexArray].name;

            return new CStatus();
        }
                
        /// <summary>
        /// US:6048 protectd helper to get the next clinic target
        /// </summary>
        /// <param name="ua"></param>
        /// <param name="strTarget"></param>
        /// <returns></returns>
        protected CStatus GetNextClinicTarget(TaggedHospitalLocationArray ha, 
                                              out string strTarget)
        {
            strTarget = String.Empty;

            if (ha.locations == null)
            {
                return new CStatus();
            }

            //get the index of the last item in the array
            int nCountArray = ha.count;
            int nIndexArray = 0;
            if (nCountArray > 0)
            {
                nIndexArray = nCountArray - 1;
            }

            //get the next target
            strTarget = ha.locations[nIndexArray].name;

            return new CStatus();
        }

        /// US:6002
        /// protected helper to get the next note title target
        /// </summary>
        /// <param name="tta"></param>
        /// <param name="strTarget"></param>
        /// <returns></returns>
        protected CStatus GetNextNoteTitleTarget(TaggedTextArray tta, out string strTarget)
        {
            strTarget = String.Empty;

            if (tta.results == null)
            {
                return new CStatus();
            }

            //get the index of the last item in the array
            int nCountArray = tta.results.Count();
            int nIndexArray = 0;
            if (nCountArray > 0)
            {
                nIndexArray = nCountArray - 1;
            }

            if(tta.results[nIndexArray].taggedResults == null)
            {
                //no results
                return new CStatus();
            }

            //get the index of the last item in the taggedResults of the array
            int nCountTagged = tta.results[nIndexArray].taggedResults.Count();
            int nIndexTagged = 0;
            if (nCountTagged > 0)
            {
                nIndexTagged = nCountTagged - 1;
            }

            //get the next target
            strTarget = tta.results[nIndexArray].taggedResults[nIndexTagged].textArray[0];

            return new CStatus();
        }

        /// <summary>
        /// US:6002
        /// gets note titles from MDWS and transfers them to the db
        /// </summary>
        /// <returns></returns>
        public CStatus TransferNoteTitles(string strMatch)
        {
            //check to make sure the MDWS connection is valid
            /*CStatus status = IsMDWSValid();
            if (!status.Status)
            {
                return false;
            }
            */
            CStatus status = new CStatus();


            //build a search string
            string strSearch = String.Empty;
            if (strMatch != null)
            {
                strSearch = strMatch.ToUpper();
            }

            //per Joel: You pass text to those calls corresponding to the 
            //name of the clinic/not title where you want to begin searching. 
            //So, say I wanted to get note titles for comp and pen notes, 
            //I would enter “comp” or something like that. 
            //The list appears longer because CPRS is automagically 
            //executing the same call over and over transparently to you 
            //as you scroll through the names. By default, those RPCs 
            //usually return 44 records per search starting with the name 
            //of the argument you supplied.
            //
            //get the first batch of notes
            TaggedTextArray tta = GetMDWSSOAPClient().getNoteTitles(strSearch, "1");
            //return if tta is null
            if (tta == null || tta.fault != null)
            {
                return new CStatus(false, k_STATUS_CODE.Failed, "Failed to retrieve note titles!");
            }

            //loop and continue to get data until we have them all
            string strCurrentTarget = String.Empty;
            string strPreviousTarget = String.Empty;
            while (tta != null && tta.fault == null)
            {
                //get the next target for the search
                //if the current target = the previous target we are done
                //processing the array
                GetNextNoteTitleTarget(tta, out strCurrentTarget);
                if (strCurrentTarget == strPreviousTarget)
                {
                    break;
                }

                //transfer this block of note titles
                //when the matches come back they start with the note titles that match the search then
                //the list continues on with note titles that do not match. this means we need to check for
                //an actual match and quit transferring when we do not find one!
                CMDWSTransfer transfer = new CMDWSTransfer(this);
                status = transfer.TransferNoteTitles(tta, strSearch);
                if (!status.Status)
                {
                    return status;
                }

                //return if we are done
                if (status.StatusCode == k_STATUS_CODE.Done)
                {
                    return new CStatus();
                }

                //make the call again with the target and direction = "1" = forward
                tta = GetMDWSSOAPClient().getNoteTitles(strCurrentTarget, "1");

                //set the previous target = the current target
                strPreviousTarget = strCurrentTarget;
            }

            return new CStatus();
        }


        /// <summary>
        /// US:6005 transfers a patients appointment to the central database
        /// </summary>
        /// <returns></returns>
        public CStatus TransferPatientAppointments()
        {
            //check to make sure the MDWS connection is valid
            /*CStatus status = IsMDWSValid();
             if (!status.Status)
             {
                 return false;
             }
             */
            CStatus status = new CStatus();


            //get selected patient appointments from MDWS
            TaggedAppointmentArrays taas = GetMDWSSOAPClient().getAppointments();
            if (taas == null || taas.fault != null)
            {
                return new CStatus(false, k_STATUS_CODE.Failed, "Failed to retrieve patient appointments!");
            }

            //transfer the appointments to the central db
            CMDWSTransfer xfer = new CMDWSTransfer(this);
            status = xfer.TransferPatientAppointments(taas);

            return status;
        }
        
        /// <summary>
        /// US:6007 transfers a patients consults to the central database
        /// </summary>
        /// <returns></returns>
        public CStatus TransferPatientConsults()
        {
            //check to make sure the MDWS connection is valid
            /*CStatus status = IsMDWSValid();
            if (!status.Status)
            {
                return false;
            }
            */
            CStatus status = new CStatus();


            //get the consults for the selected patient from MDWS
            TaggedConsultArrays taas = GetMDWSSOAPClient().getConsultsForPatient();
            if (taas == null || taas.fault != null)
            {
                return new CStatus(false, k_STATUS_CODE.Failed, "Failed to retrieve patient consults!");
            }

            //transfer the consults to the central database
            CMDWSTransfer xfer = new CMDWSTransfer(this);
            status = xfer.TransferPatientConsults(taas);

            return status;
        }

        /// <summary>
        /// US:6071 transfer patient visits
        /// </summary>
        /// <returns></returns>
        public CStatus TransferPatientVisits(DateTime dtFrom, DateTime dtTo)
        {
            //check to make sure the MDWS connection is valid
            /*CStatus status = IsMDWSValid();
            if (!status.Status)
            {
                return false;
            }
            */
            CStatus status = new CStatus();


            string strFrom = CDataUtils2.GetMDWSShortDateString(dtFrom, "000000");
            string strTo = CDataUtils2.GetMDWSShortDateString(dtTo, "235959");

            TaggedVisitArray taas = GetMDWSSOAPClient().getVisits(strFrom,
                                                                  strTo);


            //per wasa scan do not return sensitive information to the user.
            if (taas == null || taas.fault != null)
            {
                return new CStatus(false, k_STATUS_CODE.Failed, "Failed to retrieve patient visits, if this error continues please contact your system administrator.");
                //return new CStatus(false, k_STATUS_CODE.Failed, "Failed to retrieve patient visits! " + taas.fault.message);
            }

            //transfer the visits to the central database
            CMDWSTransfer xfer = new CMDWSTransfer(this);
            status = xfer.TransferPatientVisits(taas);

            return status;
        }
        
        /// <summary>
        /// US:6006 transfer patient admissions to the central database
        /// </summary>
        /// <returns></returns>
        public CStatus TransferPatientAdmissions()
        {
            //check to make sure the MDWS connection is valid
            /*CStatus status = IsMDWSValid();
            if (!status.Status)
            {
                return false;
            }
            */
            CStatus status = new CStatus();


            //get the admissions
            TaggedInpatientStayArray taas = GetMDWSSOAPClient().getAdmissions();
            if (taas == null || taas.fault != null)
            {
                return new CStatus(false, k_STATUS_CODE.Failed, "Failed to retrieve patient admissions!");
            }

            //transfer the admissions for the central database
            CMDWSTransfer xfer = new CMDWSTransfer(this);
            status = xfer.TransferPatientAdmissions(taas);

            return status;
        }

        /// <summary>
        /// US:838
        /// gets teams from MDWS and moves them to the db
        /// </summary>
        /// <returns></returns>
        public CStatus GetMDWSTeams()
        {
            //count
            long lCount = 0;

            //check to make sure the MDWS connection is valid
            /*CStatus status = IsMDWSValid();
             if (!status.Status)
             {
                 return false;
             }
             */
            CStatus status = new CStatus();

            //get the teams from MDWS
            TaggedText txtTeams = GetMDWSSOAPClient().getTeams();
            if (txtTeams == null || txtTeams.fault != null)
            {
                //return new CMDWSStatus(txtTeams.fault);
                return new CStatus(false, k_STATUS_CODE.Failed, "Error retrieving MDWS teams");
            }

            //transfer the patients to the db
            CMDWSTransfer transfer = new CMDWSTransfer(this);
            status = transfer.TransferTeams(txtTeams, out lCount);
            if (!status.Status)
            {
                return status;
            }

            return new CStatus();
        }


        /// <summary>
        /// does a user have a specific key
        /// </summary>
        /// <param name="lUserID"></param>
        /// <param name="strKey"></param>
        /// <returns></returns>
        public CStatus HasSecurityKey(long lUserID,
                                      string strKey)
        {
            CStatus status = new CStatus(); 

            //get the teams from MDWS
            //get the users security keys
            UserSecurityKeyArray usk = GetMDWSSOAPClient().getUserSecurityKeys(lUserID.ToString());
            if (usk == null || usk.fault != null)
            {
                string strErr = "Error retrieving security keys, if this error continues please contact your system administrator.";
                               
                //per wasa scan do not return sensitive information to the user.
                //if(usk.fault != null)
                //{
                //    strErr += ": " + usk.fault.message;
                //}

                //return new CMDWSStatus(usk.fault);
                return new CStatus(false, k_STATUS_CODE.Failed, strErr);
            }

            //loop over the keys and look for a match
            foreach (UserSecurityKeyTO keyTO in usk.keys)
            {
                if (keyTO != null)
                {
                    if (keyTO.name.ToUpper().Trim() == strKey.ToUpper().Trim())
                    {
                        return new CStatus(true, k_STATUS_CODE.Success, "User has key");
                    }                
                }
            }

            //if we get here, no match user does not have key
            return new CStatus(true, k_STATUS_CODE.Failed, "User does not have key");
        }


        /// <summary>
        /// US:5348 gets MDWS security keys and transfers them to the database
        /// </summary>
        /// <param name="lUserID"></param>
        /// <param name="bTransfer"></param>
        /// <returns></returns>
        public CStatus GetMDWSSecurityKeys(
            long lUserID,
            bool bTransfer)
        {
            //count
            long lCount = 0;

            //todo! this is failing
            return new CStatus();

            //check to make sure the MDWS connection is valid
            /*CStatus status = IsMDWSValid();
            if (!status.Status)
            {
                return false;
            }
            */
            CStatus status = new CStatus();


            //get the teams from MDWS
            //get the users security keys
            UserSecurityKeyArray usk = GetMDWSSOAPClient().getUserSecurityKeys(lUserID.ToString());
            if (usk == null || usk.fault != null)
            {
                //return new CMDWSStatus(usk.fault);
                return new CStatus(false, k_STATUS_CODE.Failed, "Error retrieving security keys");
            }

            //in some cases we do not want to transfer the keys...
            if (bTransfer)
            {
                //transfer the patients to the db
                CMDWSTransfer transfer = new CMDWSTransfer(this);
                status = transfer.TransferSecurityKeys(
                    lUserID,
                    usk,
                    out lCount);
                if (!status.Status)
                {
                    return status;
                }
            }

            return new CStatus();
        }

        /// <summary>
        /// US:838
        /// gets specialties from MDWS and moves them to the db
        /// </summary>
        /// <returns></returns>
        public CStatus GetMDWSSpecialties()
        {
            //count
            long lCount = 0;

            //check to make sure the MDWS connection is valid
            /*CStatus status = IsMDWSValid();
            if (!status.Status)
            {
                return false;
            }
            */
            CStatus status = new CStatus();


            //get the specialties from MDWS
            TaggedText txtSpecialties = GetMDWSSOAPClient().getSpecialties();
            if (txtSpecialties == null || txtSpecialties.fault != null)
            {
                //return new CMDWSStatus(txtSpecialties.fault);
                return new CStatus(false, k_STATUS_CODE.Failed, "Error retrieving specialties");
            }

            //transfer the data to the db
            CMDWSTransfer transfer = new CMDWSTransfer(this);
            status = transfer.TransferSpecialties(txtSpecialties, out lCount);
            if (!status.Status)
            {
                return status;
            }

            return new CStatus();
        }

        /// <summary>
        /// US:838
        /// gets wards from MDWS and moves them to the db
        /// </summary>
        /// <returns></returns>
        public CStatus GetMDWSWards()
        {
            //count
            long lCount = 0;

            //check to make sure the MDWS connection is valid
            /*CStatus status = IsMDWSValid();
             if (!status.Status)
             {
                 return false;
             }
             */
            CStatus status = new CStatus();


            TaggedHospitalLocationArray thla = GetMDWSSOAPClient().getWards();
            if (thla == null || thla.fault != null)
            {
                //return new CMDWSStatus(thla.fault);
                return new CStatus(false, k_STATUS_CODE.Failed, "Error retrieving wards");
            }

            //transfer the data to the db
            CMDWSTransfer transfer = new CMDWSTransfer(this);
            status = transfer.TransferWards(thla, out lCount);
            if (!status.Status)
            {
                return status;
            }

            return new CStatus();
        }

        /// <summary>
        /// US:6048
        /// gets clinics from MDWS and moves them to the db
        /// </summary>
        /// <returns></returns>
        public CStatus TransferClinics(string strMatch)
        {
            //check to make sure the MDWS connection is valid
            /*CStatus status = IsMDWSValid();
            if (!status.Status)
            {
                return false;
            }
            */
            CStatus status = new CStatus();


            //build a search string
            string strSearch = String.Empty;
            if (strMatch != null)
            {
                strSearch = strMatch.ToUpper();
            }

            //the list continues on with users that do not match. this means we need to check for
            //an actual match and quit transferring when we do not find one!
            //

            //per Joel: You pass text to those calls corresponding to the 
            //name of the clinic/not title where you want to begin searching. 
            //So, say I wanted to get note titles for comp and pen notes, 
            //I would enter “comp” or something like that. 
            //The list appears longer because CPRS is automagically 
            //executing the same call over and over transparently to you 
            //as you scroll through the names. By default, those RPCs 
            //usually return 44 records per search starting with the name 
            //of the argument you supplied.
            //
            //get the first batch of clinics
            TaggedHospitalLocationArray thla = GetMDWSSOAPClient().getClinics(strSearch);
            if (thla == null || thla.fault != null)
            {
                //return new CMDWSStatus(thla.fault);
                return new CStatus(false, k_STATUS_CODE.Failed, "Error retrieving clinics");
            }

            //loop and continue to get data until we have them all
            string strCurrentTarget = String.Empty;
            string strPreviousTarget = String.Empty;
            while (thla != null && thla.fault == null)
            {
                //get the next target for the search
                //if the current target = the previous target we are done
                //processing the array
                GetNextClinicTarget(thla, out strCurrentTarget);
                if (strCurrentTarget == strPreviousTarget)
                {
                    break;
                }

                //transfer the data to the db
                CMDWSTransfer transfer = new CMDWSTransfer(this);
                status = transfer.TransferClinics(thla, strSearch);
                if (!status.Status)
                {
                    return status;
                }

                //return if we are done
                if (status.StatusCode == k_STATUS_CODE.Done)
                {
                    return new CStatus();
                }

                //make the call again with the new target
                thla = GetMDWSSOAPClient().getClinics(strCurrentTarget);

                //set the previous target = the current target
                strPreviousTarget = strCurrentTarget;
            }

            return new CStatus();
        }

        /// <summary>
        /// US:838
        /// transfers patients by teams to the db
        /// </summary>
        /// <param name="lTeamID"></param>
        /// <returns></returns>
        public CStatus GetMDWSTeamPatients(long lTeamID)
        {
            //count
            long lCount = 0;

            //check to make sure the MDWS connection is valid
            /*CStatus status = IsMDWSValid();
             if (!status.Status)
             {
                 return false;
             }
             */
            CStatus status = new CStatus();


            //get the teams from MDWS
            TaggedPatientArray tpa = GetMDWSSOAPClient().getPatientsByTeam(lTeamID.ToString());
            if (tpa == null || tpa.fault != null)
            {
                //return new CMDWSStatus(tpa.fault);
                return new CStatus(false, k_STATUS_CODE.Failed, "Error retrieving team patients");
            }

            //transfer the patients to the db
            CMDWSTransfer transfer = new CMDWSTransfer(this);
            status = transfer.TransferPatientArray(
                string.Empty,
                lTeamID,
                0,
                0,
                0,
                tpa,
                out lCount);
            if (!status.Status)
            {
                return status;
            }

            return new CStatus();
        }

        /// <summary>
        /// US:838
        /// tranfer MDWS patients that match criteria
        /// </summary>
        /// <param name="strMatchID"></param>
        /// <returns></returns>
        public CStatus GetMDWSMatchPatients(string strMatch)
        {
            //count
            long lCount = 0;

            //check to make sure the MDWS connection is valid
            /*CStatus status = IsMDWSValid();
            if (!status.Status)
            {
                return false;
            }
            */
            CStatus status = new CStatus();

            //get the teams from MDWS
            TaggedPatientArrays tpas = GetMDWSSOAPClient().match(strMatch);
            if (tpas == null || tpas.fault != null)
            {
                //return new CMDWSStatus(tpas.fault);
                return new CStatus(false, k_STATUS_CODE.Failed, "Error looking up patients");
            }

            foreach (TaggedPatientArray tpa in tpas.arrays)
            {
                //transfer the patients to the db
                CMDWSTransfer transfer = new CMDWSTransfer(this);
                status = transfer.TransferPatientArray(
                    string.Empty,
                    0,
                    0,
                    0,
                    0,
                    tpa,
                    out lCount);
                if (!status.Status)
                {
                    return status;
                }
            }

            return new CStatus();
        }

        /// <summary>
        /// US:838
        /// transfers patients by specialty to the db
        /// </summary>
        /// <param name="lSpecialtyID"></param>
        /// <returns></returns>
        public CStatus GetMDWSSpecialtyPatients(long lSpecialtyID)
        {
            //count
            long lCount = 0;

            //check to make sure the MDWS connection is valid
            /*CStatus status = IsMDWSValid();
             if (!status.Status)
             {
                 return false;
             }
             */
            CStatus status = new CStatus();


            //get the teams from MDWS
            TaggedPatientArray tpa = GetMDWSSOAPClient().getPatientsBySpecialty(lSpecialtyID.ToString());
            if (tpa == null || tpa.fault != null)
            {
                //return new CMDWSStatus(tpa.fault);
                return new CStatus(false, k_STATUS_CODE.Failed, "Error retrieving specialty patients");
            }

            //transfer the patients to the db
            CMDWSTransfer transfer = new CMDWSTransfer(this);
            status = transfer.TransferPatientArray(
                string.Empty,
                0,
                lSpecialtyID,
                0,
                0,
                tpa,
                out lCount);
            if (!status.Status)
            {
                return status;
            }

            return new CStatus();
        }

        /// <summary>
        /// US:838
        /// transfers patients by ward to the db
        /// </summary>
        /// <param name="lWardID"></param>
        /// <returns></returns>
        public CStatus GetMDWSWardPatients(long lWardID)
        {
            //count
            long lCount = 0;

            //check to make sure the MDWS connection is valid
            /*CStatus status = IsMDWSValid();
             if (!status.Status)
             {
                 return false;
             }
             */
            CStatus status = new CStatus();


            //get the wards from MDWS
            TaggedPatientArray tpa = GetMDWSSOAPClient().getPatientsByWard(lWardID.ToString());
            if (tpa == null || tpa.fault != null)
            {
                //return new CMDWSStatus(tpa.fault);
                return new CStatus(false, k_STATUS_CODE.Failed, "Error retrieving ward patients");
            }

            //transfer the patients to the DB
            CMDWSTransfer transfer = new CMDWSTransfer(this);
            status = transfer.TransferPatientArray(
                string.Empty,
                0,
                0,
                lWardID,
                0,
                tpa,
                out lCount);
            if (!status.Status)
            {
                return status;
            }

            return new CStatus();
        }

        /// <summary>
        /// US:838
        /// transfers patients by clinic to the db
        /// </summary>
        /// <param name="lClinicID"></param>
        /// <param name="dtApptFrom"></param>
        /// <param name="dtApptTo"></param>
        /// <param name="lCount"></param>
        /// <returns></returns>
        public CStatus GetMDWSClinicPatients(
            long lClinicID,
            DateTime dtApptFrom,
            DateTime dtApptTo)
        {
            //count
            long lCount = 0;

            //check to make sure the MDWS connection is valid
            /*CStatus status = IsMDWSValid();
            if (!status.Status)
            {
                return status;
            }*/
            CStatus status = new CStatus();


            //get the wards from MDWS
            TaggedPatientArray tpa = GetMDWSSOAPClient().getPatientsByClinic(Convert.ToString(lClinicID));
            if (tpa == null || tpa.fault != null)
            {
                //return new CMDWSStatus(tpa.fault);
                return new CStatus(false, k_STATUS_CODE.Failed, "Error retrieving clinic patients");
            }

            //transfer the patients to the DB
            CMDWSTransfer transfer = new CMDWSTransfer(this);

            status = transfer.TransferPatientArray(
                string.Empty,
                0,
                0,
                0,
                lClinicID,
                tpa,
                out lCount);
            if (!status.Status)
            {
                return status;
            }

            return new CStatus();
        }

        /// <summary>
        /// US:6047 transfers users containing the specified search string
        /// </summary>
        /// <param name="strSearch"></param>
        /// <returns></returns>
        public CStatus TransferUsers(string strMatch)
        {
            //check to make sure the MDWS connection is valid
            /*CStatus status = IsMDWSValid();
             if (!status.Status)
             {
                 return false;
             }
             */
            CStatus status = new CStatus();


            //build a search string
            string strSearch = String.Empty;
            if (strMatch != null)
            {
                strSearch = strMatch.ToUpper();
            }


            //when the matches come back they start with the users that match the search then
            //the list continues on with users that do not match. this means we need to check for
            //an actual match and quit transferring when we do not find one!
            //

            //per Joel: You pass text to those calls corresponding to the 
            //name of the clinic/not title where you want to begin searching. 
            //So, say I wanted to get note titles for comp and pen notes, 
            //I would enter “comp” or something like that. 
            //The list appears longer because CPRS is automagically 
            //executing the same call over and over transparently to you 
            //as you scroll through the names. By default, those RPCs 
            //usually return 44 records per search starting with the name 
            //of the argument you supplied.
            //
            //get the first batch of users
        
            UserArray ua = GetMDWSSOAPClient().cprsUserLookup(strSearch);
            if (ua == null || ua.fault != null)
            {
                //return CMDWSStatus(ua.fault);
                return new CStatus(false, k_STATUS_CODE.Failed, "Error retrieving MDWS users");
            }
            
            //loop and continue to get data until we have them all
            string strCurrentTarget = String.Empty;
            string strPreviousTarget = String.Empty;
            while (ua != null && ua.fault == null)
            {
                //get the next target for the search
                //if the current target = the previous target we are done
                //processing the array
                GetNextUserTarget(ua, out strCurrentTarget);
                if (strCurrentTarget == strPreviousTarget)
                {
                    break;
                }

                //transfer the users to the central db
                CMDWSTransfer transfer = new CMDWSTransfer(this);
                status = transfer.TransferUserArray(ua, strSearch);
                if (!status.Status)
                {
                    return status;
                }

                //return if we are done
                if (status.StatusCode == k_STATUS_CODE.Done)
                {
                    return new CStatus();
                }

                //make the call again with the new target
                ua = GetMDWSSOAPClient().cprsUserLookup(strCurrentTarget);
               
                //set the previous target = the current target
                strPreviousTarget = strCurrentTarget;
            }


            return new CStatus();

        }

        /// <summary>
        /// US:838
        /// retrieves a list of the users patients and transfers 
        /// them to the db
        /// </summary>
        /// <param name="lUserID"></param>
        /// <returns></returns>
        public CStatus GetMDWSUserPatients(long lUserID)
        {
            //count
            long lCount = 0;

            //check to make sure the MDWS connection is valid
            /*CStatus status = IsMDWSValid();
            if (!status.Status)
            {
                return false;
            }
            */
            CStatus status = new CStatus();


            //get the patients from MDWS
            string strDUZ = Convert.ToString(lUserID);
            TaggedPatientArray tpa = GetMDWSSOAPClient().getPatientsByProvider(strDUZ);
            if (tpa == null || tpa.fault != null)
            {
                //return new CMDWSStatus(tpa.fault);
                return new CStatus(false, k_STATUS_CODE.Failed, "Error retrieving user patients");
            }

            //transfer the patients to the db
            CMDWSTransfer transfer = new CMDWSTransfer(this);
            status = transfer.TransferPatientArray(
                strDUZ,
                0,
                0,
                0,
                0,
                tpa,
                out lCount);
            if (!status.Status)
            {
                return status;
            }

            return new CStatus();
        }

        /// <summary>
        /// another version of login that takes additional params 
        /// used from the communicator and and clients that store sitelist and 
        /// context in places other than a config file
        /// </summary>
        /// <param name="strUN"></param>
        /// <param name="strPA"></param>
        /// <param name="strSiteList"></param>
        /// <param name="strContext"></param>
        /// <param name="toUser"></param>
        /// <returns></returns>
        protected CStatus Login(
            string strUN,
            string strPA,
            string strSiteList,
            string strContext,
            out UserTO toUser)
        {
            toUser = null;

            try
            {
                //connect to MDWS
                DataSourceArray dsa = GetMDWSSOAPClient().connect(strSiteList);
                if (dsa == null || dsa.fault != null && dsa.fault.message != "You are already connected to that site")
                {
                    //per wasa scan do not return sensitive information to the user.
                    //return new CMDWSStatus(dsa.fault);
                    return new CStatus(false, k_STATUS_CODE.Failed, "Already connected");
                }

                //login to mdws
                toUser = GetMDWSSOAPClient().login(
                    strUN,
                    strPA,
                    strContext);
                if (toUser == null || toUser.fault != null)
                {
                    //return new CMDWSStatus(toUser.fault);
                    return new CStatus(false, k_STATUS_CODE.Failed, "Unable to login to MDWS, please check your credentials! " + toUser.fault.message);
                }
            }
            catch (Exception e)
            {
                return new CStatus(false, k_STATUS_CODE.Failed, e.Message);
            }

            return new CStatus();
        }

        /// <summary>
        /// US:5348 
        /// login to MDWS and transfer user to the db
        /// </summary>
        /// <param name="strUID"></param>
        /// <param name="strPWD"></param>
        /// <param name="lUserID"></param>
        /// <param name="mdwsSOAPClient"></param>
        /// <returns></returns>
        public CStatus MDWSLogin(string strUID,
                                  string strPWD,
                                  long lRegionID,
                                  long lSiteID,
                                  string strContext,
                                  out string strDUZ,
                                  out long lFXUserID,
                                  out EmrSvcSoapClient mdwsSOAPClient)
        {
            //outs
            lFXUserID = -1;
            strDUZ = string.Empty;
            mdwsSOAPClient = null;
            UserTO toUser = null;

            //mdws soap client
            try
            {
                mdwsSOAPClient = new EmrSvcSoapClient("EmrSvcSoap");
            }
            catch (Exception e)
            {
                mdwsSOAPClient = null;
                return new CStatus(false, k_STATUS_CODE.Failed, e.Message);
            }

            //initialize sitelist and context
            string strSiteList = Convert.ToString(lSiteID);
            try
            {
                //disconnect first...
                mdwsSOAPClient.disconnect();

                //connect to MDWS
                DataSourceArray dsa = mdwsSOAPClient.connect(strSiteList);

                //failed so try again
                if (dsa == null)
                {
                    //failed so disconnect and try again
                    mdwsSOAPClient.disconnect();

                    //reconnect
                    dsa = mdwsSOAPClient.connect(strSiteList);
                    if (dsa == null)
                    {
                        return new CStatus(false, k_STATUS_CODE.Failed, "An undefined error occurred while connecting to MDWS!");
                    }
                }

                //failed with fault so try again if already connected
                if (dsa.fault != null)
                {
                    if (dsa.fault.message != "You are already connected to that site")
                    {
                        string strErr = "Error connecting to site, if this error continues please contact your system administrator.";

                        //per wasa scan do not return sensitive information to the user.
                        //return new CMDWSStatus(dsa.fault);
                        return new CStatus(false, k_STATUS_CODE.Failed, strErr);
                    }
                    else
                    {
                        //try again disconnect
                        mdwsSOAPClient.disconnect();

                        //try to re-connect
                        dsa = mdwsSOAPClient.connect(strSiteList);
                        if (dsa == null)
                        {
                            return new CStatus(false, k_STATUS_CODE.Failed, "An undefined error occurred while connecting to MDWS!");
                        }
                        else
                        {
                            if (dsa.fault != null)
                            {
                                //per wasa scan do not return sensitive information to the user.
                                //return new CMDWSStatus(dsa.fault);
                                return new CStatus(false, k_STATUS_CODE.Failed, "An undefined error occurred while connecting to MDWS!");
                   
                            }
                        }
                    }
                }

                //login to mdws
                toUser = mdwsSOAPClient.login(strUID,
                                              strPWD,
                                              strContext);

                //check for errors
                if (toUser == null)
                {
                    return new CStatus(false,
                                       k_STATUS_CODE.Failed,
                                       "Unable to login to MDWS, please check your credentials! ");
                }
                else
                {
                    if (toUser.fault != null)
                    {
                        //per wasa scan do not return sensitive information to the user.
                        //but in this case we do want to show the standard A/V failed message
                        //return new CMDWSStatus(toUser.fault);
                        if (toUser.fault.message.Trim() == "Not a valid ACCESS CODE/VERIFY CODE pair.")
                        {
                            return new CStatus(false, k_STATUS_CODE.Failed, "Not a valid ACCESS CODE/VERIFY CODE pair.");
                        }
                        else
                        {
                            string strErr = "Unable to login to MDWS, please check your credentials. " + toUser.fault.message;
                            return new CStatus(false, k_STATUS_CODE.Failed, strErr);
                        }
                    }
                }

            }
            catch (Exception e)
            {
                //per wasa scan do not return sensitive information to the user.
                //return new CStatus(false, k_STATUS_CODE.Failed, e.Message);
                string strErr = "Unable to login to MDWS, please check your credentials. " + e.Message;
                return new CStatus(false, k_STATUS_CODE.Failed, strErr);
                
            }

            //keep the duz
            strDUZ = toUser.DUZ;

            //transfer the user data to the db
            //this will return the central db's user id
            CMDWSTransfer transfer = new CMDWSTransfer(this);
            CStatus status = transfer.TransferUser(toUser, lRegionID, lSiteID, out lFXUserID);
            if (!status.Status)
            {
                mdwsSOAPClient.disconnect();
                mdwsSOAPClient = null;
                return status;
            }

            //note: BaseMster.MDWSEmrSvcClient gets cached in session state
            //if the login was successful

            //if we get here we are logged in
            return new CStatus();
        }

        /// <summary>
        /// US:1952 US:1945 US:885
        /// write a note to MDWS
        /// </summary>
        /// <param name="strPatientID"></param>
        /// <param name="strAuthorID"></param>
        /// <param name="strVisitLocationID"></param>
        /// <param name="strNoteTitleIEN"></param>
        /// <param name="strNoteText"></param>
        /// <returns></returns>
        public CStatus WriteNote(
            string strPatientID,
            string strAuthorID,
            string strESignature,
            string strVisitLocationID,
            string strNoteTitleIEN,
            string strNoteText)
        {
            //select the patient
            PatientTO toSelect = GetMDWSSOAPClient().select(strPatientID);

            // easy way to make sure we're using new value each test run -- yes 
            //this will create a lot of notes...
            DateTime dtnow = DateTime.Now;
            string strVisitTimeStamp = dtnow.ToString("yyyyMMdd.HHmmss");

            //encounter
            // create our encounter string
            // use "A" not a historical note, "H" if it was...
            string strEncounter = strVisitLocationID + ";" + strVisitTimeStamp + ";H";

            string strCosigner = String.Empty;
            string strConsultIEN = String.Empty;
            string strPrfIEN = String.Empty;

            //before we start any note writing we need to verify the 
            //esignature so that we dont have dangling notes if they 
            //enter a bad esig one time
            TextTO toTxt = GetMDWSSOAPClient().isValidEsig(strESignature);
            if (toTxt.text == null || toTxt.text.ToUpper() != "TRUE")
            {
                return new CStatus(false,
                                    k_STATUS_CODE.Failed,
                                    "Invalid Signature Code!");
            }

            //write the note
            //
            //For preserving line breaks within a text block that is 
            //being written to a note, replace the \r\n or \n characters 
            //with a pipe (|).
            //
            NoteResultTO noteResult = GetMDWSSOAPClient().writeNote(
                strNoteTitleIEN,
                strEncounter,
                strNoteText.Replace("\r\n", "|"),
                strAuthorID,
                strCosigner,
                strConsultIEN,
                strPrfIEN);

            //check for error while writing note...
            if (noteResult == null || noteResult.fault != null)
            {
                //per wasa scan do not return sensitive information to the user.
                //return new CMDWSStatus(noteResult.fault);
                string strErr = "An error occurred while writing the note, ";
                strErr += "if this error continues please contact your system administrator.";
                return new CStatus(false, k_STATUS_CODE.Failed, strErr);
  
            }

            //sign the note
            TextTO toSign = GetMDWSSOAPClient().signNote(noteResult.id,
                                                         strAuthorID,
                                                         strESignature);

            //check for sign error and return
            if (toSign.fault != null)
            {
                //per wasa scan do not return sensitive information to the user.
                //return new CMDWSStatus(toSign.fault);
                string strErr = "An error occurred while signing the note, ";
                strErr += "if this error continues please contact your system administrator.";
                return new CStatus(false, k_STATUS_CODE.Failed, strErr);
            }

            //close the note
            TextTO closeNoteResult = GetMDWSSOAPClient().closeNote(noteResult.id, string.Empty);

            //check for close error and return
            if (closeNoteResult == null || closeNoteResult.fault != null)
            {
                //per wasa scan do not return sensitive information to the user.
                //return new CMDWSStatus(closeNoteResult.fault);
                string strErr = "An error occurred while writing the note, ";
                strErr += "if this error continues please contact your system administrator.";
                return new CStatus(false, k_STATUS_CODE.Failed, strErr);
            }

           
            return new CStatus();
        }

        /// <summary>
        /// write a note with no signing, left open for user to sign later in CPRS
        /// </summary>
        /// <param name="strPatientID"></param>
        /// <param name="strAuthorID"></param>
        /// <param name="strVisitLocationID"></param>
        /// <param name="strNoteTitleIEN"></param>
        /// <param name="strNoteText"></param>
        /// <param name="strVisitTimeStamp"></param>
        /// <param name="strConsultIEN"></param>
        /// <param name="strCosigner"></param>
        /// <param name="strAH"></param>
        /// <returns></returns>
        public CStatus WriteNote2(
           string strPatientID,
           string strAuthorID,
           string strNoteTitleIEN,
           string strEncounterID,//created and passed in
           string strNoteText,
           string strConsultIEN,
           string strCosignerDUZ,
           out string strTIUNoteID,
           out string strTIUNote) //A or H note
        {
            strTIUNoteID = "";
            strTIUNote = "";

            //prf ien
            string strPrfIEN = String.Empty;

            //write the note, patient is alreadyd selected at this point
            //
            //For preserving line breaks within a text block that is 
            //being written to a note, replace the \r\n or \n characters 
            //with a pipe (|).
            //
            NoteResultTO noteResult = GetMDWSSOAPClient().writeNote(strNoteTitleIEN,
                                                                    strEncounterID,
                                                                    strNoteText.Replace("\r\n", "|"),
                                                                    strAuthorID,
                                                                    strCosignerDUZ,
                                                                    strConsultIEN,
                                                                    strPrfIEN);

            //check for error while writing note...
            if (noteResult == null || noteResult.fault != null)
            {
                //per wasa scan do not return sensitive information to the user.
                //return new CMDWSStatus(noteResult.fault);

                string strErr = "An error occurred while writing the note, ";
                strErr += "if this error continues please contact your system administrator.";

                if (noteResult.fault != null)
                {
                    if (noteResult.fault.message.Length < 60)
                    {
                        strErr += " " + noteResult.fault.message;
                    }
                }

                return new CStatus(false, k_STATUS_CODE.Failed, strErr);
            }

            //this prevents the 'Another session is editing this entry' error from CPRS
            TextTO note2 = GetMDWSSOAPClient().closeNote(noteResult.id, "");


            //Get the note just written
            TextTO note = GetMDWSSOAPClient().getNote(Convert.ToString(this.SiteID),
                                                      noteResult.id);
            if (note == null || note.fault != null)
            {
                //per wasa scan do not return sensitive information to the user.
                //return new CMDWSStatus(noteResult.fault);
                string strErr = "An error occurred while writing the note, ";
                strErr += "if this error continues please contact your system administrator.";
                if (noteResult.fault != null)
                {
                    if (noteResult.fault.message.Length < 60)
                    {
                        strErr += " " + noteResult.fault.message;
                    }
                }

                return new CStatus(false, k_STATUS_CODE.Failed, strErr);
            }

            //pass the note text back
            strTIUNoteID = noteResult.id;
            strTIUNote = note.text;

            //good to go
            return new CStatus();
        }

        /// <summary>
        /// US:852
        /// transfers static lab test information from mdws to the vappct database
        /// </summary>
        /// <param name="strSearch"></param>
        /// <returns></returns>
        public CStatus GetMDWSLabTests(string strSearch)
        {
            if (String.IsNullOrEmpty(strSearch))
            {
                return new CStatus(false, k_STATUS_CODE.Failed, "Search is empty");
            }

            //check to make sure the MDWS connection is valid
            /*CStatus status = IsMDWSValid();
            if (!status.Status)
            {
                return false;
            }
            */
            CStatus status = new CStatus();


            string strNextSearch = strSearch.Substring(0, 1).ToUpper();
            string strSrch = strNextSearch;
            while (true)
            {
                TaggedLabTestArrays la = GetMDWSSOAPClient().getLabTests(strSrch);
                if (la == null || la.fault != null)
                {
                    //return new CMDWSStatus(la.fault);
                    return new CStatus(false, k_STATUS_CODE.Failed, "Error searching");
                }

                //transfer the lab tests to the db
                long lCount = 0;
                CMDWSTransfer transfer = new CMDWSTransfer(this);
                status = transfer.TransferLabTestArray(
                    la,
                    out lCount,
                    out strNextSearch);
                if (!status.Status)
                {
                    return status;
                }

                //check for break
                if (String.IsNullOrEmpty(strNextSearch))
                {
                    break;
                }
                if (strNextSearch.Substring(0, 1).ToUpper() != strSearch.Substring(0, 1).ToUpper())
                {
                    break;
                }

                strSrch = strNextSearch;
            }

            return status;
        }
    }
